using System;
using System.IO;
using System.Xml;
using gov.va.med.VBECS.Communication.Common;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// Base class for all VistALink protocol messages.
	/// Derived from common message interface (CR 3304)
	/// </summary>
	public abstract class VistALinkMessage : IMessage
	{
		private static readonly VersionNumber VistALinkVersion = new VersionNumber( "1.0" );

		/// <summary>
		/// VistALink message type. For example 'gov.va.med.foundations.rpc.request'.
		/// </summary>
		protected readonly string _messageType; 

		/// <summary>
		/// Name of XSD schema corresponding to VistALink message XML
		/// </summary>

		protected readonly string _sourceXSD;

		/// <summary>
		/// Message handler type. May be 'single call' for stateless protocol implementations 
		/// (or specific messages that does not require response / indicate that connection must be closed
		/// after processing a request), or 'singleton' for stateful implementations.
		/// </summary>
		protected HandlerMode _handlerMode;

		// Constants used in XML message serialization/deserialization		
		private const string XMLCONSTS_VISTALINK_NODE_NAME = "VistaLink";
		private const string XMLCONSTS_VERSION_VL_ATTRIBUTE_NAME = "version";		
		private const string XMLCONSTS_MESSAGE_TYPE_VL_ATTRIBUTE_NAME = "messageType";
		private const string XMLCONSTS_XMLNS_VL_ATTRIBUTE_NAME = "xmlns";
		private const string XMLCONSTS_XMLNS_VL_ATTRIBUTE_VALUE = "http://med.DNS   /Foundations";
		private const string XMLCONSTS_XSD_XSI_SCHEMA_LOCATION_VL_ATTRIBUTE_NAME = "xsi:noNamespaceSchemaLocation";
		private const string XMLCONSTS_XMLNS_XSI_VL_ATTRIBUTE_NAME = "xmlns:xsi";
		private const string XMLCONSTS_XMLNS_XSI_VL_ATTRIBUTE_VALUE = "http://www.w3.org/2001/XMLSchema-instance";
		private const string XMLCONSTS_HANDLER_MODE_ATTRIBUTE_NAME = "mode";

		/// <summary>
		/// Empty static constructor added to get rid of "beforefieldinit" attribute generated by compiler.
		/// </summary>
		static VistALinkMessage() {}

		/// <summary>
		/// The complete constructor that should be used by derived classes to 
		/// fully specify parameters implemented in VistALinkMessage class.  
		/// All parameters are required.
		/// </summary>
		/// <param name="messageType">VistALink message type.</param>
		/// <param name="sourceXSD">
		///		VistALink message-bound XSD file name (i.e. rpcRequest.xsd). 
		///		It's assumed that the specified XSD is available in assembly resources.
		/// </param>
		protected VistALinkMessage( string messageType, string sourceXSD )
		{
            Id = Guid.NewGuid().ToString();

			if( messageType == null )
				throw( new ArgumentNullException( "messageType" ) );

			if( sourceXSD == null )
				throw( new ArgumentNullException( "sourceXSD" ) );
			
			_messageType = messageType; 
			_sourceXSD = sourceXSD;
			_handlerMode = HandlerMode.Singleton;
		}

		/// <summary>
		/// Standard XML deserialization constructor. Allows recreating 
		/// message object from supplied XML document. This constructor should be used by child classes that 
		/// want to be deserializable. It will parse and verify parameters common for all VistALink messages.
		/// It will also validate message against corresponding XSD schema if validation is enabled in 
		/// global configuration. 
		/// </summary>
		/// <param name="messageType">VistALink message type.</param>
		/// <param name="sourceXSD">
		///		VistALink message-bound XSD file name (i.e. rpcRequest.xsd). 
		///		It's assumed that the specified XSD is available in assembly resources.
		/// </param>
		/// <param name="sourceDoc">Source XML document to parse and use as a source.</param>
		protected VistALinkMessage( string messageType, string sourceXSD, XmlDocument sourceDoc ) 
			: this( messageType, sourceXSD )
		{		
			if( sourceDoc == null ) 
				throw( new ArgumentNullException( "sourceDoc" ) );

			if( sourceDoc.PreserveWhitespace )
				throw( new ArgumentException( SR.Exceptions.VistALinkMessageSourceXmlDocPreservesWhiteSpace() ) );

			XmlElement _docElement = XmlParseSafeGetEnvelopeRootElement( sourceDoc );

			XmlUtility.ParseCheckRequiredAttributeValue( _docElement, XMLCONSTS_XMLNS_XSI_VL_ATTRIBUTE_NAME, XMLCONSTS_XMLNS_XSI_VL_ATTRIBUTE_VALUE );

			// Checking for XML namespace attribute is disabled on Foundations team 
			// request - Java XML parsers do not handle it correctly, so M server does not include this attribute anymore. 
			// However, due to this restriction, all XML messages returned by server as-is do not conform to XSDs.
			// XmlParseCheckRequiredAttributeValue( _docElement, XMLCONSTS_XMLNS_VL_ATTRIBUTE_NAME, XMLCONSTS_XMLNS_VL_ATTRIBUTE_VALUE );

			XmlUtility.ParseCheckRequiredAttributeValue( _docElement, XMLCONSTS_XSD_XSI_SCHEMA_LOCATION_VL_ATTRIBUTE_NAME, _sourceXSD );

			XmlUtility.ParseCheckRequiredAttributeValue( _docElement, XMLCONSTS_MESSAGE_TYPE_VL_ATTRIBUTE_NAME, _messageType );

			XmlUtility.ParseCheckRequiredAttribute( _docElement, XMLCONSTS_VERSION_VL_ATTRIBUTE_NAME );

			// Message VistALink protocol version must be equal or greater than the supported version
			if( VistALinkVersion.CompareTo( VersionNumber.Parse( _docElement.GetAttribute( XMLCONSTS_VERSION_VL_ATTRIBUTE_NAME ) ) ) > 0 )
				throw( new XmlParseException( SR.Exceptions.VistaALinkProtocolVersionMismatch( 
					VistALinkVersion.ToString(), _docElement.GetAttribute( XMLCONSTS_VERSION_VL_ATTRIBUTE_NAME ) ) ) );

			// Handler mode defaults to singleton and is optional
			if( _docElement.HasAttribute( XMLCONSTS_HANDLER_MODE_ATTRIBUTE_NAME ) )
				_handlerMode = HandlerMode.Parse( _docElement.GetAttribute( XMLCONSTS_HANDLER_MODE_ATTRIBUTE_NAME ) );
		}

		/// <summary>
		/// Writes opening part of VistALink XML message envelope
		/// </summary>
		/// <param name="writer">Open XmlWriter to write to.</param>
		private void WriteStartXmlEnvelope( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartDocument();

			writer.WriteStartElement( XMLCONSTS_VISTALINK_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_MESSAGE_TYPE_VL_ATTRIBUTE_NAME, _messageType );
			writer.WriteAttributeString( XMLCONSTS_VERSION_VL_ATTRIBUTE_NAME, VistALinkVersion.ToString() );						
			writer.WriteAttributeString( XMLCONSTS_HANDLER_MODE_ATTRIBUTE_NAME, _handlerMode.ToString() );
			writer.WriteAttributeString( XMLCONSTS_XMLNS_XSI_VL_ATTRIBUTE_NAME, XMLCONSTS_XMLNS_XSI_VL_ATTRIBUTE_VALUE );
			writer.WriteAttributeString( XMLCONSTS_XSD_XSI_SCHEMA_LOCATION_VL_ATTRIBUTE_NAME, _sourceXSD );
			writer.WriteAttributeString( XMLCONSTS_XMLNS_VL_ATTRIBUTE_NAME, XMLCONSTS_XMLNS_VL_ATTRIBUTE_VALUE );
		}

		/// <summary>
		/// Writes closing part of VistALink XML message envelope
		/// </summary>
		/// <param name="writer">Open XmlWriter to write to</param>
		private void WriteEndXmlEnvelope( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteEndElement();

			writer.WriteEndDocument();
		}

		/// <summary>
		/// Writes XML message body (serializes object to 
		/// the corresponding VistALink protocol XML message).		
		/// Child classes must implement this method to write theirs specific message bodies.
		/// </summary>
		/// <param name="writer">Open XmlWriter to use</param>
		protected abstract void WriteXmlMessageBody( XmlWriter writer );

		/// <summary>
		/// Serializes message to XML format with specified indentation and packs it to byte array.
		/// </summary>
		/// <param name="formatting">XML indentation to</param>
		/// <returns>XML representation of the message, packed to byte array</returns>
        private byte[] ToXmlByteArray(Formatting formatting)
		{
			MemoryStream _stream = new MemoryStream();
			XmlTextWriter _writer = new XmlTextWriter( _stream, GlobalConfig.NetworkMessageEncoding );				
			_writer.Formatting = formatting;

			WriteStartXmlEnvelope( _writer );
			WriteXmlMessageBody( _writer );
			WriteEndXmlEnvelope( _writer );

			_writer.Flush();

			return _stream.ToArray();
		}

		/// <summary>
		/// Serializes message and returns it as an XML string. Provides human-readable well-indented XML. 		
		/// </summary>
		/// <remarks>
		///		This method may be highly ineffective. It should be only used for debug and unit testing purposes.  
		///		More efficient implementation may be created in future releases. 
		/// </remarks>
		/// <returns>Indented XML representation of the message</returns>
		public string ToXmlString()
		{
			return GlobalConfig.NetworkMessageEncoding.GetString( ToXmlByteArray( Formatting.Indented ) );
		}

		/// <summary>
		/// This method allows overriding handler mode for close socket request 
		/// message. It was not put in base class constructor because
		/// close socket request message is the only one requiring "single call" mode.
		/// </summary>
		/// <param name="handlerMode">
		///		Handler mode indicating how message should be handled (either as a single
		///		request/response call, or as a part of established session - singleton).
		/// </param>
		protected void SetHandlerMode( HandlerMode handlerMode )
		{
			if( handlerMode  == null )
				throw( new ArgumentNullException( "handlerMode" ) );

			_handlerMode = handlerMode;
		}

		/// <summary>
		/// This method is used during XML deserialization to recognize VistALink message subclasses.
		/// It extracts and returns message type string name from supplied XML document.
		/// </summary>
		/// <param name="sourceDoc">Source XML document to extract message type from.</param>
		/// <returns>String representing XML type of VistALink message stored in XML.</returns>
		public static string XmlParseGetVistALinkMessageTypeString( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );

			return XmlUtility.ParseGetRequiredAttributeValue( XmlParseSafeGetEnvelopeRootElement( sourceDoc ), XMLCONSTS_MESSAGE_TYPE_VL_ATTRIBUTE_NAME );
		}

		/// <summary>
		/// Utility XML parse method. It gets reference to root message element performing all necessary checks.
		/// </summary>
		/// <param name="sourceDoc">Source XML document to work with.</param>
		/// <returns>Reference to root message element.</returns>
		private static XmlElement XmlParseSafeGetEnvelopeRootElement( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );
	
			if( sourceDoc.DocumentElement == null )
				throw( new XmlParseException( SR.Exceptions.VistALinkMessageXmlDocumentElementNotFound( XMLCONSTS_VISTALINK_NODE_NAME ) ) );

			if( sourceDoc.DocumentElement.Name != XMLCONSTS_VISTALINK_NODE_NAME )
				throw( new XmlParseException( 
					SR.Exceptions.VlMessageFormatIncorrectDocumentElement( XMLCONSTS_VISTALINK_NODE_NAME, sourceDoc.DocumentElement.Name ) ) ) ;

			return sourceDoc.DocumentElement;
		}

		/// <summary>
		/// Property exposing message handler mode - either a singleton or a single call. 
		/// </summary>
		public HandlerMode HandlerMode
		{
			get
			{
				return _handlerMode;
			}
		}

        #region IMessage methods

        /// <summary>
        /// Message ID
        /// </summary>
        public string Id { get; set; }
	    /// <summary>
	    /// Message Reply ID (assigned if it is a reply message)
	    /// </summary>
	    public string RepliedId { get; set; }

        /// <summary>
        /// Serializes message to XML format without indentation and packs it to byte array.
        /// </summary>
        /// <returns>XML representation of the message, packed to byte array.</returns>
        public byte[] GetBytes()
        {
            return ToXmlByteArray(Formatting.None);
        }

        #endregion
    }
}
